"""
Recursive Audio Metadata Tagger (FIXED WAV HANDLING)
---------------------------------------------------

Written by ChatGPT at the command of the user.

Features:
- Recursively scans current directory and subdirectories
- Smart MP3 / WAV detection
- Auto-derives TITLE from filename
- User-specified cover image filename
- Website + PayPal stored separately (best practice)
- MP3 cover art embedding (APIC)
- WAV RIFF INFO written correctly (no Frame errors)
- Preserves existing metadata
- Skips already-tagged files
- Safe for large archives

Requirements:
    pip install mutagen
"""

import os
import re
from mutagen.mp3 import MP3
from mutagen.id3 import (
    ID3, TIT2, TPE1, TALB, TDRC,
    COMM, TCON, TXXX, APIC, WOAR, WXXX
)
from mutagen.wave import WAVE

AUDIO_EXTS = (".mp3", ".wav")

# ------------------------------------------------------------
# SMART TITLE FROM FILENAME
# ------------------------------------------------------------

def smart_title_from_filename(path):
    name = os.path.splitext(os.path.basename(path))[0]
    name = re.sub(r"^[\[\(]?\d{1,3}[\]\)]?\s*[-_.]?\s*", "", name)
    name = re.sub(r"[_\-.]+", " ", name)

    junk = {
        "final", "mix", "master", "demo",
        "edit", "render", "export", "bounce"
    }

    words = [w for w in name.split() if w.lower() not in junk]
    title = " ".join(words).strip().title()
    return title or "Untitled"

# ------------------------------------------------------------
# COVER ART (MP3 ONLY)
# ------------------------------------------------------------

def find_user_cover(folder, image_name):
    if not image_name:
        return None
    path = os.path.join(folder, image_name)
    return path if os.path.isfile(path) else None

def embed_cover_art_mp3(audio, image_path):
    with open(image_path, "rb") as f:
        data = f.read()

    mime = "image/png" if image_path.lower().endswith(".png") else "image/jpeg"

    audio.tags.delall("APIC")
    audio.tags.add(APIC(
        encoding=3,
        mime=mime,
        type=3,  # Front cover
        desc="Cover",
        data=data
    ))

# ------------------------------------------------------------
# USER INPUT
# ------------------------------------------------------------

def prompt_metadata():
    print("\nShared metadata (TITLE auto-generated):\n")
    return {
        "artist": input("Artist: ").strip(),
        "album": input("Album: ").strip(),
        "year": input("Year (YYYY): ").strip(),
        "genre": input("Genre: ").strip(),
        "website": input("Website URL: ").strip(),
        "paypal": input("PayPal Donation URL: ").strip(),
        "comment": input("License / Comment text: ").strip(),
        "cover_image": input("Cover image filename (e.g. cover.png): ").strip(),
        "tagged_flag": "Tagged by batch_audio_metadata_recursive.py"
    }

# ------------------------------------------------------------
# MP3 TAGGING
# ------------------------------------------------------------

def tag_mp3(path, meta):
    audio = MP3(path, ID3=ID3)

    if audio.tags is None:
        audio.add_tags()

    if "TXXX:Tagged" in audio.tags:
        return False

    audio.tags.add(TIT2(encoding=3, text=smart_title_from_filename(path)))

    if meta["artist"]:
        audio.tags.add(TPE1(encoding=3, text=meta["artist"]))
    if meta["album"]:
        audio.tags.add(TALB(encoding=3, text=meta["album"]))
    if meta["year"]:
        audio.tags.add(TDRC(encoding=3, text=meta["year"]))
    if meta["genre"]:
        audio.tags.add(TCON(encoding=3, text=meta["genre"]))

    if meta["website"]:
        audio.tags.add(WOAR(encoding=3, url=meta["website"]))

    if meta["paypal"]:
        audio.tags.add(WXXX(encoding=3, desc="PayPal", url=meta["paypal"]))
        audio.tags.add(TXXX(encoding=3, desc="PayPal", text=meta["paypal"]))

    if meta["comment"]:
        audio.tags.add(COMM(
            encoding=3,
            lang="eng",
            desc="Comment",
            text=meta["comment"]
        ))

    cover = find_user_cover(os.path.dirname(path), meta["cover_image"])
    if cover:
        embed_cover_art_mp3(audio, cover)

    audio.tags.add(TXXX(encoding=3, desc="Tagged", text=meta["tagged_flag"]))
    audio.save()
    return True

# ------------------------------------------------------------
# WAV TAGGING (CORRECT MUTAGEN USAGE)
# ------------------------------------------------------------

def tag_wav(path, meta):
    audio = WAVE(path)

    # Prevent re-tagging
    if "Tagged" in audio:
        return False

    # RIFF INFO fields MUST be assigned directly
    audio["INAM"] = smart_title_from_filename(path)

    if meta["artist"]:
        audio["IART"] = meta["artist"]
    if meta["album"]:
        audio["IPRD"] = meta["album"]
    if meta["year"]:
        audio["ICRD"] = meta["year"]
    if meta["genre"]:
        audio["IGNR"] = meta["genre"]
    if meta["website"]:
        audio["IURL"] = meta["website"]

    # WAV has no donation URL field → comment only
    comment_lines = []
    if meta["comment"]:
        comment_lines.append(meta["comment"])
    if meta["paypal"]:
        comment_lines.append(f"Donate: {meta['paypal']}")

    if comment_lines:
        audio["ICMT"] = "\n".join(comment_lines)

    # Internal marker
    audio["Tagged"] = meta["tagged_flag"]

    audio.save()
    return True

# ------------------------------------------------------------
# MAIN
# ------------------------------------------------------------

def main():
    meta = prompt_metadata()
    tagged = skipped = 0

    print("\nScanning audio files...\n")

    for root, _, files in os.walk("."):
        for f in files:
            if not f.lower().endswith(AUDIO_EXTS):
                continue

            path = os.path.join(root, f)
            try:
                changed = tag_mp3(path, meta) if f.lower().endswith(".mp3") else tag_wav(path, meta)
                if changed:
                    tagged += 1
                    print(f"✔ Tagged: {path}")
                else:
                    skipped += 1
                    print(f"↪ Skipped: {path}")
            except Exception as e:
                print(f"⚠ Error: {path} → {e}")

    print("\n✅ Done")
    print(f"Tagged: {tagged}")
    print(f"Skipped: {skipped}")

if __name__ == "__main__":
    main()
